#----------------------------------------------------------------------------#
#                                                                            #
#   pIPC is (c) 2000-2002 Fionn Behrens <fi@nn.org> and may be distributed   #
#    and used freely under the licensing terms of the GNU Public License     #
#                             (http://www.fsf.org/)                          #
#                                                                            #
#   For further information, updates and documentation of IPC.py check out:  #
#                       http://rtfm.n3.net/software/pIPC/                    #
#                                                                            #
#   If you use this software in a project that is deployed in professional   #
#  environments and/or on a large scale, you should feel obliged to send an  #
# EMail to the author (see top for address), telling him about your project. #
#  Donations and chocolate are always welcome but, certainly, not mandatory. #
#                                                                            #
#----------------------------------------------------------------------------#

__version__ = (0,9,7)

"""
pIPC                                           (c) 2000-2002 by Fionn Behrens
Released under the terms of the GPL.
This file implements the classes Pipe and PipeEnd that
can be used for hassle-free, queued inter-process communications (IPC).

Example Usage:

import pIPC

pipe = pIPC.Pipe()

start_new_thread(my_func,(pipe,my_args,...))

each process or thread can now (EQALLY!) use the pipe, f.e.:

r,w,x = select([pipe],[],[],1)

if r != []:
  if r[0] != pipe:
    raise IOError, 'unknown file IO in select()' # whatever... shouldnt happen
  data = pipe.read()    # This data is from the other thread, it must
                        # have called pipe.write(data)
  if len(pipe) > 0:           # If there are more messages than just one, you
    moredata = pipe.pending() # can retrieve an array of all pending messages
else:                         # from the pipe with the pending() method.
  pipe.write("Got no data! Are you sleeping?")  # Will be readable by the other
                                                # thread, using pipe.read()

Or whatever. The pipe class will take care of everything, including queueing
of multiple messages until pending messages are completely read.

If you want to do IPC with completely separate processes (i.e. spawn()ed) you
will have to use named pipes aka "fifo"s. Use them almost exactly like above
standard pipes, just use

pipe = pIPC.Pipe("mypipe")   # Use your favorite unique name instead of mypipe

in both processes involved. After that you can use them in just the same way
that is shown above. The fifos will be unlinked from /tmp when both sides have
opened them and remain invisble.

But that is not all. If you want to do IPC between two separate computers
through the net, just use

pipe = pIPC.Pipe("remote_host_name", port [, timeout])

on both computers involved to get your pIPC message interface as usual.
Whichever side is first to do so will open the port and the other side
will connect to it. Optionally you can specify a how long the first
link partner will wait for the second one to connect (timeout).

CAVEATS:

* Thoroughly tested only with Linux threads (pipes) and processes (fifos)

* Version 0.9.5 and up dont work with python 1.5.2 anymore.

* Communication is strictly two-way. No "one sender and two recipients" method.
  Just use two Pipe()s for stuff like that.

* Only standard Pipes are reusable. Named pipes and Network pipes will have
  to be disposed whenever the connection has been interrupted. (just make
  yourself a new pair'o'Pipe()s then, its free!)

* The only thing that you will have to take care of is that if you end a thread
  and you want to re-use its end of a pipe for another process, the exiting 
  thread will have to call pipe.disconnect() before it ends. The other end of
  the pipe will get a message containing the string "BYE" (plus any string you
  can submit as an argument as disconnect(string)). Sending any further
  message after disconnect() will reconnect the pipe to the calling process.

* Messages are separated internally using the sequence "<E0X\0>". Although it
  is extremely improbable that this sequence occurs in any data stream,
  it may cause unexpected behaviour should this ever happen.

* If you have lots of data sent, use select() or do pipe.flush() regularly in
  the receiving process, or messages will overflow the posix pipe buffers and
  cause write errors on the sending thread's side. pipe.flush() will NOT
  delete messages, just flush the "physical" posix pipe and store the data
  for later retrieval via pipe.read()

* Always call the close() method of a pipe if you do not plan to use it any more
  or you might run out of file descriptors quickly.

* pIPC.py is still in development. Internal structures may change in future
  releases, although the Pipe Class API in its current form will probably stay
  as it is or be expanded, but not changed. Use of other internal structures
  is depreciated at the moment.

  Send bug reports and enhancement requests/patches to pIPC@software.fionn.de
  If this module made your day, ask for my adress and send chocolate. ;-)
"""

import os

try:
  import thread
except:
  class t_stub:
    def __init__(self):
      self.get_ident        = os.getpid
  thread = t_stub()

from fcntl   import fcntl
try:                                                        # for python 2.2+
  from fcntl import F_SETFD, F_SETFL
  from os    import O_NONBLOCK, O_NDELAY
except:                                                     # up to python 2.1
  from FCNTL import O_NDELAY, O_NONBLOCK, F_SETFD, F_SETFL
from os      import O_TRUNC, O_RDWR
from string  import atoi, find, join
from socket  import socket, AF_INET, SOCK_STREAM, gethostname
from select  import select

pdebug = 0

#----------------------------------------------------------------------------

class Pipe:

  #--------------------------------------------------
  # What follows here is the usual set of standard methods for python objects.

  def __init__(self, name=None, port=None, timeout=0):
    self.psize = 8192
    self.end   = {}
    self.name  = name
    self.port  = port              # named pipes and networked pipes do not
    if name:                       # share anything. there is only one end per
      self.ends = [ PipeEnd(name, port, timeout) ]  # process or host, the other
      self.ends.append(self.ends[0].other)          # one is "fake"
      self.ends[0].w_buf += "N%s<E0X\0>" % self.ends[0].id  # send init msg
    else:
      self.ends = [ PipeEnd(), PipeEnd() ]
      self.ends[0].connect(self.ends[1])
    self.flag  = None

  def __del__(self):
    if self.ends:
      self.disconnect()
      for end in self.ends:
        end.close()
      self.ends = []

  def __len__(self):
    pe = self.get_my_end()
    self.get_data(pe)
    return len(pe.queue)

  def __repr__(self):
    self.flush()
    desc  = ["<IPC Pipe at '%s', " % hex(id(self))[2:]]
    count = 0
    for end in self.ends:
      if end.id:
        count = count + 1
        self.get_data(end)
    desc.append(["unbound.", "bound to process", "connecting"][count])
    if count > 0:
      desc.append(" %s" % (self.ends[0].id or self.ends[1].id))
      if count > 1:
        desc.append(" and %s" % self.ends[1].id)
        count = len(self.ends[0].queue) + len(self.ends[1].queue)
        if count > 0:
          desc.append(", with %i queued message%s"%(count, ["s",""][count==1]))
      else:
        desc.append(" [not connected]")
    return(join(desc + [">"],""))

  #--------------------------------------------------
  # flush pending messages from the posix pipe into the internal
  # buffer array structure.

  def flush(self):
    pe = self.get_my_end()
    self.get_data(pe)

  connect = flush

  #--------------------------------------------------
  # like flush(), but returns the buffer array immediately

  def pending(self):
    pe = self.get_my_end()
    self.get_data(pe)
    out = pe.queue
    pe.queue = []
    return out

  #--------------------------------------------------
  # this comes in handy if a function receives an IPC that it cannot evaluate.
  # it can requeue messages into the buffer array for later retrieval.

  def requeue(self,data):
    (self.get_my_end()).queue.append(data)

  #--------------------------------------------------
  # read one single IPC message from the internal buffer queue.

  def read(self):
    pe = self.get_my_end()
    self.get_data(pe)
    if pe.queue == []:
      debug("IPC.Pipe zero read.")
      return ""
    out = pe.queue[0]
    del pe.queue[0]
    debug("pe.queue: "+repr(pe.queue))
    return out

  #--------------------------------------------------
  # private method for "physical" reading from the posix pipe.

  def get_data(self, pe):
    if pe.w_buf:
      try:
        pe.write(pe.w_buf) 
        pe.w_buf = ""
      except:
        pass
    zero_reads = 0
    while zero_reads < 3:
      x = find(pe.r_buf, "<E0X\0>")
      if x > -1:
        out = pe.r_buf[:x]
        pe.r_buf = pe.r_buf[x+6:]
        if out[0] == "D":
          pe.queue.append(out[1:])
        elif out[0] in ["X", "N"]:
          pid = [None, out[1:]][out[0]=="N"]
          if self.name:
            self.ends[1].id = pid
          else:
            for end in self.ends:
              if end != pe:
                if pid and pid not in self.end.keys():
                  self.end[pid] = end
                else:
                  self.end = { pe.id:pe }
                end.id = pid
            if self.ends[0].id == self.ends[1].id:
              debug("IPC.Pipe: Pipe ID error!! (%s)" % out)
              raise IOError, "You CAN'T use both ends of this pipe in the same thread or process!"
      else:
        try:
          data = pe.read()
          if data:
            pe.r_buf   += data
          else:
            zero_reads += 1
        except:
          break

  #--------------------------------------------------
  # use this to make sure your pipe is connected to the calling process
  # and to some other process as well.
  # CAVEAT: to connect your process to a pipe, you MUST use it in some
  #         way, or else this function will always return 0
  #         ( try pipe.connect() or pipe.flush() )

  def is_connected(self):
    try:
      myend = self.get_my_end()
      self.get_data(myend)
      if myend.other.id != None: 
        return 1
    except:
      pass
    return 0

  #--------------------------------------------------
  # write an IPC message into the pipe

  def write(self,data):
    pe = self.get_my_end()
    try:
      pe.write("%sD%s<E0X\0>" % (pe.w_buf, data))
      pe.w_buf = ""
    except:
      debug("IPC.Pipe write failed: buffer overflow?")

  #--------------------------------------------------
  # private. do not use. this function might disappear in future
  # releases and is only there for testing purposes.

  def set_psize(self,size):
    for obj in self.end.values()+[self]:
      obj.psize = size

  #--------------------------------------------------
  # just there for completeness. normally it is not necessary to
  # makefile() an IPC.Pipe() because you can use it in select()
  # directly.

  def makefile(self):
    (self.get_my_end()).makefile()
    return (self)

  #--------------------------------------------------
  # necessary for select()

  def fileno(self):
    pe = self.get_my_end()
    try:
      if not pe.fh:
        pe.makefile()
      return pe.fh.fileno()
    except:
      debug("ENDS: "+repr(self.end)+":"+repr(self.ends)+":"+pe.id)

  #--------------------------------------------------
  # whenever a thread that uses an IPC.Pipe() exits, call this function
  # FIRST, or you will not be able to use that pipe anymore.

  def disconnect(self, data=""):
    pe = self.get_my_end()
    self.get_data(pe)
    debug("IPC.Pipe: Disconnecting.")
    try:
      pe.write("X%s<E0X\0>DBYE%s<E0X\0>" % (pe.id, data))
    except:
      debug("ENDS: "+repr(self.end)+":"+repr(self.ends)+":"+pe.id)
    if pe.id in self.end.keys():
      del self.end[pe.id]
    pe.id = None
    pe.r_buf = ""
    pe.queue = []

  #--------------------------------------------------
  # private. do not use. may change in future versions.

  def get_my_end(self):
    myid = "%i-%i" % (os.getpid(), thread.get_ident())
    try:
      if self.name:
        return self.ends[0]      # Named and Net pipes dont need id treatment
      else:
        return self.end[myid]
    except:
      if not self.ends:
        raise IOError, 'I/O operation on closed pipe'
      for id in range(2):
        if self.ends[id].id == None and self.ends[not(id)].id != myid:
          self.ends[id].id = myid
          self.end[myid]   = self.ends[id]
          self.ends[id].w_buf += "N%s<E0X\0>" % myid
          debug("IPC.Pipe: Connecting.") 
        if self.ends[id].id == myid:
          return self.ends[id]
      debug("ENDS: "+repr(self.end))
      raise IOError, 'Third party tries to use an already connected pipe! '

  #--------------------------------------------------
  # this one probably does not need explanation.

  def close(self):
    self.__del__()

#----------------------------------------------------------------------------

# This can be used by anyone who wants to do all PipeEnd management by hand
# for one reason or the other...

def new_iopipe():
  a, b = PipeEnd(), PipeEnd()
  a.connect(b)
  return a,b

def del_iopipe(a=None, b=None):
  for end in [a,b]:
    if end != None:
      end.close()
      del end

#----------------------------------------------------------------------------

class PipeEnd:

  def __init__(self, name=None, port=None, timeout=0):
    self.name   = name
    self.port   = port
    self.psize  = 8192
    self.other  = None
    self.d_in   = None
    self.d_out  = None
    self.fh     = None
    self.conn   = None
    self.id     = None
    self.r_buf  = ""
    self.w_buf  = ""
    self.queue  = []
    self.open(timeout)

  def __del__(self):
    self.close()

  #--------------------------------------------------

  def open(self, timeout=0):
    if self.d_out == None:
      if not self.name:                                 #--- OS Pipes ---
        self.pipe  = os.pipe()
      elif not self.port:                               #--- Named pipes ---
        if os.path.exists("/tmp/pipc_%s_in" % self.name):
          try:
            f_out = os.open("/tmp/pipc_%s_in" % self.name, O_RDWR|O_TRUNC)
            f_in  = open("/tmp/pipc_%s_out" % self.name, "r")
          except:
            raise IOError, 'failed to open existing named pipes (%s)' %self.name
          self.other     = PipeDummy()
          self.other.cid = 2
          self.pdel()
        else:
          try:
            for direction in ["in","out"]:
              os.mkfifo("/tmp/pipc_%s_%s" % (self.name, direction))
          except:
            raise IOError, 'PIPC: failed to create fifo: %s' % self.name
          try:
            f_out = os.open("/tmp/pipc_%s_out" % self.name, O_RDWR|O_TRUNC)
            dummy = open("/tmp/pipc_%s_in" % self.name, "w+")
            f_in  = open("/tmp/pipc_%s_in" % self.name, "r")
            dummy.close()
          except:
            self.pdel()
            raise IOError, 'PIPC: failed to open named pipes (%s)' % self.name
          self.other     = PipeDummy()
          self.other.cid = 1
        self.fh   = f_in
        self.pipe = (f_in.fileno(), f_out)
        self.d_in = self.pipe[0]
        self.id   = "%s:%s" % (self.name, os.getpid())
      else:                                           #--- Networked "pipe" ---
        self.other     = PipeDummy()
        try:
          conn = socket(AF_INET, SOCK_STREAM)
        except:
          raise IOError, 'PIPC: failed to open a network socket'
        try:
          conn.connect((self.name, self.port))
          self.other.cid = 2
        except:
          try:
            conn.bind(("localhost", self.port))
            conn.listen(5)
            if timeout: r, w, x = select([conn], [], [], timeout)
            else:       r, w, x = select([conn], [], [])
            if r:
              conn = conn.accept()[0]
              self.other.cid = 1
            else:
              raise IOError 'PIPC: timeout waiting for network connection'
          except:
            raise IOError, 'PIPC: neither connection nor bind succeeded'
        self.conn = conn
        self.fh   = conn.makefile()
        self.pipe = (self.fh.fileno(), self.fh.fileno())
        self.d_in = self.pipe[0]
        self.id   = "%s:%s" % (gethostname(), os.getpid())
        
      debug("MP: opening. pipe is "+repr(self.pipe))
      for p_end in [0,1]:
        fcntl(self.pipe[p_end], F_SETFD, O_NDELAY)
        fcntl(self.pipe[p_end], F_SETFL, O_NONBLOCK)
      self.myend = self.pipe[1]
      self.yrend = self.pipe[0]
      self.d_out = self.myend

  #--------------------------------------------------

  def pdel(self):
    try:
      for direction in ["in","out"]:
        os.remove("/tmp/pipc_%s_%s" % (self.name, direction))
    except:
      pass

  #--------------------------------------------------

  def connect(self, other_end=None):
    if not self.name:
      self.d_in       = other_end.yrend
      other_end.d_in  = self.yrend
      self.other      = other_end
      other_end.other = self

  #--------------------------------------------------

  def makefile(self):
    self.check()
    if self.fh == None:
      self.fh = os.fdopen(self.d_in,"r")
    return self.fh

  #--------------------------------------------------

  def write(self, data):
    self.check()
    try:
      if self.conn:
        self.conn.send(data)
      else:
        os.write(self.d_out,data)
    except:
      debug("MP: WRITE ERROR!")

  #--------------------------------------------------

  def read(self, size = None):
    if not size: size = self.psize
    self.check()
    return os.read(self.d_in,size)

  #--------------------------------------------------

  def close(self):
    if self.d_in == None:
      return
    debug("MP: closing. pipe is "+repr(self.pipe))
    if self.port:
      self.conn.close()
      self.port  = None
      self.name  = None
      self.other = None
    elif self.name:
      if self.other.cid == 1:
        self.pdel()
      self.name  = None
      self.other = None
    if self.fh != None and not(self.fh.closed):
      try:
        self.fh.close()
      except:
        #debug("MP: close fail: "+repr(self.fh))
        pass
    for fd in self.pipe:
      try:
        os.close(fd)
      except:
        #debug("MP: close fail: "+repr(fd))
        pass
    self.d_in       = None
    if self.other != None:
      self.other.other = None
      self.other       = None

  #--------------------------------------------------

  def check(self):
    if self.other == None:
      raise IOError, 'Pipe End is not connected'

#----------------------------------------------------------------------------
# Needed as a placeholder for the "other side" in fifo (named) and
# networked pipes

class PipeDummy:

  def __init__(self):
    self.id    = None
    self.r_buf = ""
    self.w_buf = ""
    self.queue = ""

  def close(self):
    pass

#----------------------------------------------------------------------------
# everything below is just there for debugging purposes. it might or might not
# disappear in future versions.

from sys import stdout

def debug(text,nl=1):
  if not pdebug:
    return
  if nl:
    text = text + "\n"
  stdout.write(str(thread.get_ident())+" ["+str(os.getpid())+"] : "+text)
  stdout.flush()

#----------------------------------------------------------------------------
