#----------------------------------------------------------------------------#
#                                                                            #
#   pIPC is (c) 2000-2001 Fionn Behrens <fi@nn.org> and may be distributed   #
#    and used freely under the licensing terms of the GNU Public License     #
#                             (http://www.fsf.org/)                          #
#                                                                            #
#   For further information, updates and documentation of IPC.py check out:  #
#                       http://rtfm.n3.net/software/pipc/                    #
#                                                                            #
#   If you use this software in a project that is deployed in professional   #
#  environments and/or on a large scale, you should feel obliged to send an  #
# EMail to the author (see top for address), telling him about your project. #
#  Donations and chocolate are always welcome but, certainly, not mandatory. #
#                                                                            #
#----------------------------------------------------------------------------#

__version__ = (0,9,1)

"""
pIPC                                           (c) 2000-2001 by Fionn Behrens
Released under the terms of the GPL.
This file implements the classes Pipe and PipeEnd that
can be used for hassle-free, queued inter-process communications (IPC).

Example Usage:

import pIPC

pipe = pIPC.Pipe()

start_new_thread(my_func,(pipe,my_args,...))

each process or thread can now (EQALLY!) use the pipe, f.e.:

r,w,x = select([pipe],[],[],1)

if r != []:
  if r[0] != pipe:
    raise IOError, "unknown file IO in select()" # whatever... shouldnt happen
  data = pipe.read()    # This data is from the other thread, it must
                        # have called pipe.write(data)
  if len(pipe) > 0:           # If there are more messages than just one, you
    moredata = pipe.pending() # can retrieve an array of all pending messages
else:                         # from the pipe with the pending() method.
  pipe.write("Got no data! Are you sleeping?")  # Will be readable by the other
                                                # thread, using pipe.read()

Or whatever. The pipe class will take care of everything, including queueing
of multiple messages until pending messages are completely read.

CAVEATS:

* Thoroughly tested only with linux threads

* The only thing that you will have to take care of is that if you end a thread
  and you want to re-use its end of a pipe for another process, the exiting 
  thread will have to call pipe.disconnect() before it ends. The other end of
  the pipe will get a message containing the string "BYE" (plus any string you
  can submit as an argument as disconnect(string)). Sending any further
  message after disconnect() will reconnect the pipe to the calling process.

* Messages are separated internally using the sequence "<EOF\0>". Although it
  is extremely improbable that this sequence occurs in any data stream,
  it may cause unexpected behaviour if it does.

* If you have lots of data sent, use select() or do pipe.flush() regularly, or
  messages will overflow the posix pipe buffers and cause write errors on the
  sending thread's side. pipe.flush() will NOT delete messages, just flush the
  "physical" posix pipe and store the data for later retrieval via pipe.read()

* Always call the close() method of a pipe if you do not plan to use it any more
  or you might run out of file descriptors quickly.

* IPC.py is still in development. Internal structures may change in future
  releases, although the Pipe Class API in its current form will probably stay
  as it is or be expanded, but not changed. Use of other internal structures
  is depreciated at the moment.

  Send bug reports and enhancement requests/patches to python-ipc@spamfilter.de
  If this module made your day, ask for my adress and send chocolate. ;-)
"""

import os

try:
  import thread
except:
  class t_stub:
    def __init__(self):
      self.get_ident = os.getpid
  thread = t_stub()

from fcntl   import fcntl
from FCNTL   import O_NDELAY, O_NONBLOCK, F_SETFD, F_SETFL
from string  import atoi, find

pdebug = 0

#----------------------------------------------------------------------------

class Pipe:

  #--------------------------------------------------
  # What follows here is the usual set of standard methods for python objects.

  def __init__(self):
    self.psize = 8192
    self.end   = {}
    self.ends  = [ PipeEnd(), PipeEnd() ]
    self.ends[0].connect(self.ends[1])
    self.flag  = None

  def __del__(self):
    for end in self.ends:
      end.close()

  def __len__(self):
    pe = self.get_my_end()
    self.get_data(pe)
    return len(pe.queue)

  def __repr__(self):
    desc  = "<IPC Pipe at '%s', "%hex(id(self))[2:]
    count = 0
    for end in self.ends:
      if end.id:
        count = count + 1
        self.get_data(end)
    desc = desc + ["disconnected.", "bound to ", "connecting "][count]
    if count > 0:
      desc = desc + str(self.ends[0].id or self.ends[1].id)
      if count > 1:
        desc = desc + " and " + str(self.ends[1].id)
        count = len(self.ends[0].queue) + len(self.ends[1].queue)
        if count > 0:
          desc = desc + ", with " + str(count) + " queued messages"
    return(desc + ">")

  #--------------------------------------------------
  # flush pending messages from the posix pipe into the internal
  # buffer array structure.

  def flush(self):
    pe = self.get_my_end()
    self.get_data(pe)

  #--------------------------------------------------
  # like flush(), but returns the buffer array immediately

  def pending(self):
    pe = self.get_my_end()
    self.get_data(pe)
    out = pe.queue
    pe.queue = []
    return out

  #--------------------------------------------------
  # this comes in handy if a function receives an IPC that it cannot evaluate.
  # it can requeue messages into the buffer array for later retrieval.

  def requeue(self,data):
    (self.get_my_end()).queue.append(data)

  #--------------------------------------------------
  # read one single IPC message from the internal buffer queue.

  def read(self):
    pe = self.get_my_end()
    self.get_data(pe)
    if pe.queue == []:
      debug("IPC.Pipe zero read.")
      return ""
    out = pe.queue[0]
    del pe.queue[0]
    debug("pe.queue: "+repr(pe.queue))
    return out

  #--------------------------------------------------
  # private method for "physical" reading from the posix pipe.

  def get_data(self,pe):
    while 1:
      x = find(pe.r_buf, "<EOF\0>")
      if x > -1:
        out = pe.r_buf[:x]
        pe.r_buf = pe.r_buf[x+6:]
        if out[0] == "D":
          pe.queue.append(out[1:])
        elif out[0] == "X" or out[0] == "N":
            pid = atoi(out[1:])
            if out[0] == "X":
              pid = None
              self.end = {pe.id:pe}
            for end in self.ends:
              if end != pe:
                end.id = pid
            if self.ends[0].id == self.ends[1].id:
              debug("IPC.Pipe: Pipe ID error!! ("+sig+";"+data+")")
      else:
        try:
          pe.r_buf = pe.r_buf + pe.read()
        except:
          break

  #--------------------------------------------------
  # use this to make sure your pipe is connected to the calling process
  # and to some other process as well.
  # CAVEAT: to connect your process to a pipe, you MUST use it in some
  #         way, or else this function will always return 0

  def is_connected(self):
    try:
      myend = self.end[thread.get_ident()]
      if myend.other.id != None: 
        return 1
    except:
      pass
    return 0

  #--------------------------------------------------
  # write an IPC message into the pipe

  def write(self,data):
    pe = self.get_my_end()
    try:
      pe.write(pe.w_buf+"D"+data+"<EOF\0>")
      pe.w_buf = ""
    except:
      debug("IPC.Pipe write failed: buffer overflow?")

  #--------------------------------------------------
  # private. do not use. this function might disappear in future
  # releases and is only there for testing purposes.

  def set_psize(self,size):
    for obj in self.end.values()+[self]:
      obj.psize = size

  #--------------------------------------------------
  # just there for completeness. normally it is not necessary to
  # makefile() an IPC.Pipe() because you can use it in select()
  # directly.

  def makefile(self):
    (self.get_my_end()).makefile()
    return (self)

  #--------------------------------------------------
  # necessary for select()

  def fileno(self):
    pe = self.get_my_end()
    if not pe.fh:
      pe.makefile()
    return pe.fh.fileno()

  #--------------------------------------------------
  # whenever a thread that uses an IPC.Pipe() exits, call this function
  # FIRST, or you will not be able to use that pipe anymore.

  def disconnect(self, data=""):
    pe = self.get_my_end()
    self.get_data(pe)
    debug("IPC.Pipe: Disconnecting.") 
    del self.end[pe.id]
    pe.write("X"+str(pe.id)+"<EOF\0>DBYE"+data+"<EOF\0>")
    pe.id = None
    pe.r_buf = ""
    pe.queue = []

  #--------------------------------------------------
  # private. do not use. may change in future versions.

  def get_my_end(self):
    try:
      return self.end[thread.get_ident()]
    except:
      myid = thread.get_ident()
      for id in range(2):
        if self.ends[id].id == None and self.ends[not(id)].id != myid:
          self.ends[id].id = myid
          self.end[myid]   = self.ends[id]
          self.ends[id].w_buf = self.ends[id].w_buf + "N"+str(myid)+"<EOF\0>"
          debug("IPC.Pipe: Connecting.") 
        if self.ends[id].id == myid:
          return self.ends[id]
      debug("ENDS: "+repr(self.end))
      print myid,"tried to use ",repr(self),"\t",repr(self.ends)
      raise IOError, 'Third party tries to use an already connected pipe! '

  #--------------------------------------------------
  # this one probably does not need explanation.

  def close(self):
    self.__del__()

#----------------------------------------------------------------------------

# This can be used by anyone who wants to do all PipeEnd management by hand
# for one reason or the other...

def new_iopipe():
  a, b = PipeEnd(), PipeEnd()
  a.connect(b)
  return a,b

def del_iopipe(a=None, b=None):
  for end in [a,b]:
    if end != None:
      end.close()
      del end

#----------------------------------------------------------------------------

class PipeEnd:

  def __init__(self):
    self.psize  = 8192
    self.other  = None
    self.d_in   = None
    self.d_out  = None
    self.fh     = None
    self.id     = None
    self.r_buf  = ""
    self.w_buf  = ""
    self.queue  = []
    self.open()

  def __del__(self):
    self.close()

  #--------------------------------------------------

  def open(self):
    if self.d_out == None:
      self.pipe  = os.pipe()
      debug("MP: opening. pipe is "+repr(self.pipe))
      fcntl(self.pipe[0], F_SETFD, O_NDELAY)
      fcntl(self.pipe[0], F_SETFL, O_NONBLOCK)
      fcntl(self.pipe[1], F_SETFD, O_NDELAY)
      fcntl(self.pipe[1], F_SETFL, O_NONBLOCK)
      self.myend = self.pipe[1]
      self.yrend = self.pipe[0]
      self.d_out = self.myend

  #--------------------------------------------------

  def connect(self, other_end):
    self.d_in       = other_end.yrend
    other_end.d_in  = self.yrend
    self.other      = other_end
    other_end.other = self

  #--------------------------------------------------

  def makefile(self):
    self.check()
    if self.fh == None:
      self.fh = os.fdopen(self.d_in,"r")
    return self.fh

  #--------------------------------------------------

  def write(self, data):
    self.check()
    try:
      os.write(self.d_out,data)
    except:
      debug("MP: WRITE ERROR!")

  #--------------------------------------------------

  def read(self, size = None):
    if not size: size = self.psize
    self.check()
    return os.read(self.d_in,size)

  #--------------------------------------------------

  def close(self):
    if self.d_in == None:
      return
    debug("MP: closing. pipe is "+repr(self.pipe))
    if self.fh != None and not(self.fh.closed):
      try:
        self.fh.close()
      except:
        #debug("MP: close fail: "+repr(self.fh))
        pass
    for fd in self.pipe:
      try:
        os.close(fd)
      except:
        #debug("MP: close fail: "+repr(fd))
        pass
    self.d_in       = None
    if self.other != None:
      self.other.other = None
      self.other       = None

  #--------------------------------------------------

  def check(self):
    if self.other == None:
      raise IOerror, 'Pipe End is not connected'

#----------------------------------------------------------------------------
# everything below is just there for debugging purposes. it might or might not
# disappear in future versions.

from sys import stdout

def debug(text,nl=1):
  if not pdebug:
    return
  if nl:
    text = text + "\n"
  stdout.write(str(thread.get_ident())+" ["+str(os.getpid())+"] : "+text)
  stdout.flush()

#----------------------------------------------------------------------------
